import sys
import datetime
import subprocess
import configparser
import psutil
import os
from PyQt5.QtWidgets import QApplication, QMainWindow, QWidget, QPushButton, QVBoxLayout, QHBoxLayout, QTabWidget, \
    QToolButton, QDialog, QLabel, QLineEdit, QDialogButtonBox, QFileDialog, QMessageBox, QTextEdit, QSizePolicy
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import Qt, QThread, pyqtSignal, QDateTime


def resource_path(relative_path):
    """ 获取资源文件的绝对路径 """
    if getattr(sys, 'frozen', False):
        # 如果是打包后的应用，使用 sys._MEIPASS
        base_path = sys._MEIPASS
    else:
        # 如果是未打包的脚本，使用当前工作目录
        base_path = os.path.abspath(".")
    return os.path.join(base_path, relative_path)
    





class EditDialog(QDialog):
    def __init__(self, button_name, script_path, parent=None):
        super(EditDialog, self).__init__(parent)
        self.setWindowTitle("编辑按钮")
        self.button_name = button_name
        self.script_path = script_path

        # 创建布局
        layout = QVBoxLayout()

        # 创建按钮名称编辑框
        self.name_label = QLabel("按钮名称:", self)
        self.name_edit = QLineEdit(button_name, self)
        layout.addWidget(self.name_label)
        layout.addWidget(self.name_edit)

        # 创建脚本路径编辑框和浏览按钮
        self.path_label = QLabel("程序路径:", self)
        self.path_edit = QLineEdit(script_path, self)
        self.browse_btn = QPushButton("浏览", self)
        self.browse_btn.clicked.connect(self.browse_file)
        path_layout = QHBoxLayout()
        path_layout.addWidget(self.path_edit)
        path_layout.addWidget(self.browse_btn)
        layout.addWidget(self.path_label)
        layout.addLayout(path_layout)

        # 创建按钮框
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)

        # 将按钮框添加到布局中
        layout.addWidget(self.button_box)
        self.setLayout(layout)

    def browse_file(self):
        file_path, _ = QFileDialog.getOpenFileName(self, "选择程序路径", "", "所有文件 (*)")
        if file_path:
            self.path_edit.setText(file_path)

    def get_data(self):
        return self.name_edit.text(), self.path_edit.text()


class RenameTabDialog(QDialog):
    def __init__(self, current_name, parent=None):
        super(RenameTabDialog, self).__init__(parent)
        self.setWindowTitle("重命名选项卡")
        self.current_name = current_name

        # 创建布局
        layout = QVBoxLayout()

        # 创建选项卡名称编辑框
        self.name_label = QLabel("选项卡名称:", self)
        self.name_edit = QLineEdit(current_name, self)
        layout.addWidget(self.name_label)
        layout.addWidget(self.name_edit)

        # 创建按钮框
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)

        # 将按钮框添加到布局中
        layout.addWidget(self.button_box)
        self.setLayout(layout)

    def get_data(self):
        return self.name_edit.text()


class RunScriptThread(QThread):
    output_signal = pyqtSignal(str, int)  # 添加索引参数
    error_signal = pyqtSignal(str, int)   # 添加索引参数
    finished_signal = pyqtSignal(int)     # 添加索引参数

    def __init__(self, script, log_file_path, index, parent=None):
        super(RunScriptThread, self).__init__(parent)
        self.script = script
        self.log_file_path = log_file_path
        self.process = None
        self.index = index

    def run(self):
        try:
            # 使用进程组来启动进程
            self.process = subprocess.Popen(self.script, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                            preexec_fn=os.setsid)
            self.error_signal.emit(f"进程启动，PID: {self.process.pid}", self.index)

            with open(self.log_file_path, 'a') as log_file:
                # 追加当前日期和时间
                current_time = QDateTime.currentDateTime().toString("yyyy-MM-dd hh:mm:ss")
                log_file.write(f"-----------{current_time}-----------\n")

                while True:
                    output = self.process.stdout.readline().decode('utf-8', errors='ignore')
                    error = self.process.stderr.readline().decode('utf-8', errors='ignore')

                    if output == '' and error == '' and self.process.poll() is not None:
                        break
                    if output:
                        log_file.write(output)
                        self.output_signal.emit(output.strip(), self.index)
                    if error:
                        log_file.write(error)
                        self.error_signal.emit(error.strip(), self.index)

            rc = self.process.poll()
            if rc == -15:
                self.error_signal.emit(f"进程被终止，返回码 {rc}", self.index)
            elif rc is None:
                self.error_signal.emit(f"进程状态未知，返回码 {rc}", self.index)
            elif rc != 0:
                self.error_signal.emit(f"命令失败，返回码 {rc}", self.index)
            else:
                self.output_signal.emit("命令执行成功", self.index)
        except Exception as e:
            self.error_signal.emit(f"运行命令时发生错误: {str(e)}", self.index)
        finally:
            self.finished_signal.emit(self.index)

    def stop(self):
        if self.process and self.process.poll() is None:
            try:
                # 使用 psutil 来终止进程及其子进程
                proc = psutil.Process(self.process.pid)
                self.error_signal.emit(f"尝试终止进程 {proc.pid} 及其子进程...", self.index)

                # 终止整个进程组
                for child in proc.children(recursive=True):
                    self.error_signal.emit(f"终止子进程 {child.pid}", self.index)
                    child.terminate()
                proc.terminate()

                # 等待进程终止
                proc.wait(timeout=5)  # 等待5秒以确保进程终止

                if proc.is_running():
                    self.error_signal.emit(f"终止进程 {proc.pid} 超时，尝试强制终止...", self.index)
                    for child in proc.children(recursive=True):
                        self.error_signal.emit(f"强制终止子进程 {child.pid}", self.index)
                        child.kill()
                    proc.kill()

                    proc.wait(timeout=5)  # 再次等待5秒以确保进程终止

                self.error_signal.emit("进程及其子进程已终止", self.index)
            except subprocess.TimeoutExpired as e:
                self.error_signal.emit(f"终止进程超时: {str(e)}", self.index)
            except psutil.NoSuchProcess:
                self.error_signal.emit(f"进程 {self.process.pid} 已不存在", self.index)
            except Exception as e:
                self.error_signal.emit(f"终止进程时发生错误: {str(e)}", self.index)


class ButtonPanel(QWidget):
    def __init__(self, tab_name, parent=None):
        super(ButtonPanel, self).__init__(parent)
        self.tab_name = tab_name

        # 读取配置文件
        self.config = configparser.ConfigParser()
        self.config_file = resource_path("conf.ini")
        self.config.read(self.config_file)

        # 如果选项卡不在配置文件中，添加它
        if tab_name not in self.config:
            self.config[tab_name] = {}

        # 创建主布局
        main_layout = QHBoxLayout()
        self.setLayout(main_layout)

        # 创建按钮区域布局
        button_area_layout = QVBoxLayout()
        button_area_widget = QWidget()
        button_area_widget.setLayout(button_area_layout)

        # 定义按钮样式表
        button_style = """
        QPushButton, QToolButton {
            border: 1px solid #8f8f91;
            border-radius: 4px;
            background-color: #ffffff;
        }
        QPushButton:hover, QToolButton:hover {
            border: 1px solid #6c6c70;
            background-color: #f0f0f0;
        }
        QPushButton:pressed, QToolButton:pressed {
            border: 1px solid #4a4a4f;
            background-color: #d0d0d0;
        }
        """

        # 创建10个按钮及其对应的编辑图标按钮和停止按钮
        self.buttons = []
        self.threads = {}
        self.logs = [[] for _ in range(10)]  # 存储每个按钮的日志信息
        for i in range(10):
            btn_layout = QHBoxLayout()

            # 创建序列号标签，使用带圈的数字
            circled_numbers = ["①", "②", "③", "④", "⑤", "⑥", "⑦", "⑧", "⑨", "⑩"]
            seq_label = QLabel(circled_numbers[i], self)
            seq_label.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
            seq_label.setFixedWidth(20)  # 设置序列号标签的固定宽度为20像素

            # 从配置文件中读取按钮名称和脚本路径
            btn_name = self.config.get(self.tab_name, f"button{i + 1}_name", fallback=f"按钮{i + 1}")
            script_path = self.config.get(self.tab_name, f"button{i + 1}_path", fallback="echo '按钮被点击了'")

            # 创建按钮名称标签
            name_label = QLabel(btn_name, self)
            name_label.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
            name_label.setFixedWidth(150)  # 设置名称标签的固定宽度为150像素（大约15个字符）
            name_label.setStyleSheet("margin: 0px;")  # 设置间距为0字符距离

            # 创建启动按钮（图标）
            run_icon = QIcon(resource_path(f"icons/start.png"))  # 使用自定义的启动图标
            run_btn = QPushButton(run_icon, "", self)
            run_btn.setFixedSize(32, 32)  # 设置按钮的固定大小为32x32像素
            run_btn.setStyleSheet(button_style)
            run_btn.clicked.connect(lambda checked=False, index=i: self.run_shell_script(index))  # 传递按钮的索引

            # 创建停止按钮（图标）
            stop_icon = QIcon(resource_path(f"icons/stop.png"))  # 使用自定义的停止图标
            stop_btn = QPushButton(stop_icon, "", self)
            stop_btn.setFixedSize(32, 32)  # 设置按钮的固定大小为32x32像素
            stop_btn.setStyleSheet(button_style)
            stop_btn.clicked.connect(lambda checked=False, index=i: self.stop_script(index))  # 传递按钮的索引
            stop_btn.setEnabled(False)  # 初始状态禁用停止按钮

            # 创建编辑图标按钮
            edit_icon = QIcon(resource_path(f"icons/edit.png"))  # 使用自定义的编辑图标
            edit_btn = QToolButton(self)
            edit_btn.setIcon(edit_icon)
            edit_btn.setFixedSize(32, 32)  # 设置按钮的固定大小为32x32像素
            edit_btn.setStyleSheet(button_style)
            edit_btn.clicked.connect(lambda checked=False, index=i: self.edit_button(index))  # 传递按钮的索引

            # 创建查看日志按钮
            log_icon = QIcon(resource_path(f"icons/look.png"))  # 使用自定义的查看日志图标
            log_btn = QToolButton(self)
            log_btn.setIcon(log_icon)
            log_btn.setFixedSize(32, 32)  # 设置按钮的固定大小为32x32像素
            log_btn.setStyleSheet(button_style)
            log_btn.clicked.connect(lambda checked=False, index=i: self.view_log(index))  # 传递按钮的索引

            # 设置按钮和名称标签之间的间距
            btn_layout.setSpacing(5)  # 调整间距为5像素（更小的间距）

            # 将序列号标签、名称标签、启动按钮、停止按钮、编辑按钮和查看日志按钮添加到水平布局中
            btn_layout.addWidget(seq_label)
            btn_layout.addWidget(name_label)
            btn_layout.addWidget(run_btn)
            btn_layout.addWidget(stop_btn)
            btn_layout.addWidget(edit_btn)
            btn_layout.addWidget(log_btn)

            # 将水平布局添加到垂直布局中
            button_area_layout.addLayout(btn_layout)

            # 创建日志文件路径
            log_file_path = resource_path(os.path.join("logs", f"{self.tab_name}_button{i + 1}.log"))

            # 确保logs目录存在
            os.makedirs(os.path.dirname(log_file_path), exist_ok=True)

            # 将按钮、脚本路径、名称标签和日志文件路径添加到列表中
            self.buttons.append((run_btn, stop_btn, script_path, name_label, log_file_path))
            self.threads[i] = None

        # 创建日志输出窗口
        self.log_text_edit = QTextEdit(self)
        self.log_text_edit.setReadOnly(True)
        self.log_text_edit.setFixedWidth(500)  # 设置日志窗口的固定宽度

        # 设置日志窗口的高度策略
        size_policy = QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)
        self.log_text_edit.setSizePolicy(size_policy)

        # 将按钮区域和日志窗口添加到主布局中
        main_layout.addWidget(button_area_widget)
        main_layout.addWidget(self.log_text_edit)

    def run_shell_script(self, index):
        run_btn, stop_btn, script_path, name_label, log_file_path = self.buttons[index]
        self.logs[index].clear()  # 清空当前按钮的日志
        self.log_text_edit.clear()
        self.log_text_edit.append(f"Tab: {self.tab_name}, 「 {name_label.text()} 」")

        # 切换图标并禁用启动按钮
        run_btn.setIcon(QIcon(resource_path(f"icons/running.png")))
        stop_btn.setEnabled(True)  # 启用停止按钮
        run_btn.setEnabled(False)  # 禁用启动按钮

        # 在日志文件中追加5行空行和当前日期时间
        with open(log_file_path, 'a') as log_file:
            # 追加5行空行
            log_file.write("\n" * 5)
            # 追加当前日期和时间
            current_time = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            log_file.write(f"-----------{current_time}-----------\n")

        # 使用系统默认方式打开文件或在后台运行脚本
        if script_path.endswith('.sh'):
            command = f'sh "{script_path}"'
        elif os.name == 'nt':  # Windows
            command = f'"{script_path}"'
        elif sys.platform == 'darwin':  # macOS
            command = f'open "{script_path}"'
        else:  # Linux 或其他类Unix系统
            command = f'xdg-open "{script_path}"'

        # 创建并启动线程
        self.thread = RunScriptThread(command, log_file_path, index)
        self.thread.output_signal.connect(self.append_output)
        self.thread.error_signal.connect(self.append_error)
        self.thread.finished_signal.connect(lambda index=index: self.on_script_finished(index))
        self.thread.start()
        self.threads[index] = self.thread

    def stop_script(self, index):
        run_btn, stop_btn, script, name_label, _ = self.buttons[index]
        thread = self.threads[index]

        if thread and thread.process:
            if thread.process.poll() is None:
                self.log_text_edit.append(f"Tab: {self.tab_name}, Button: {name_label.text()}, 停止执行命令...")
                thread.stop()
                # 重置按钮状态
                run_btn.setIcon(QIcon(resource_path(f"icons/start.png")))
                stop_btn.setEnabled(False)  # 禁用停止按钮
                run_btn.setEnabled(True)  # 重新启用启动按钮
            else:
                self.log_text_edit.append(f"Tab: {self.tab_name}, Button: {name_label.text()}, 进程已经结束")
        else:
            self.log_text_edit.append(f"Tab: {self.tab_name}, Button: {name_label.text()}, 没有找到进程")

    def append_output(self, output, index):
        self.logs[index].append(output)  # 存储输出到对应按钮的日志
        self.log_text_edit.append(output)

    def append_error(self, error, index):
        self.logs[index].append(f"Stat: {error}")  # 存储错误到对应按钮的日志
        self.log_text_edit.append(f"Stat: {error}")

    def on_script_finished(self, index):
        run_btn, stop_btn, script, name_label, _ = self.buttons[index]
        # 切换图标为结束图标并重新启用启动按钮
        run_btn.setIcon(QIcon(resource_path(f"icons/start.png")))
        stop_btn.setEnabled(False)  # 禁用停止按钮
        run_btn.setEnabled(True)  # 重新启用启动按钮

    def edit_button(self, index):
        run_btn, stop_btn, script_path, name_label, _ = self.buttons[index]
        # 如果脚本正在运行，禁用编辑按钮
        if self.threads[index] and self.threads[index].process and self.threads[index].process.poll() is None:
            QMessageBox.warning(self, "警告", f"Tab: {self.tab_name}, Button: {name_label.text()} 正在运行，无法编辑。")
            return

        dialog = EditDialog(name_label.text(), script_path, self)

        if dialog.exec_() == QDialog.Accepted:
            new_name, new_script_path = dialog.get_data()
            # 更新名称标签
            name_label.setText(new_name)
            # 更新按钮和脚本路径
            log_file_path = self.buttons[index][4]  # 获取现有的日志文件路径
            self.buttons[index] = (
                run_btn, stop_btn, new_script_path, name_label, log_file_path)
            # 保存到配置文件
            self.config.set(self.tab_name, f"button{index + 1}_name", new_name)
            self.config.set(self.tab_name, f"button{index + 1}_path", new_script_path)
            with open(self.config_file, 'w') as configfile:
                self.config.write(configfile)

            # 切换图标为启动图标
            run_btn.setIcon(QIcon(resource_path(f"icons/start.png")))

    def view_log(self, index):
        _, _, _, name_label, log_file_path = self.buttons[index]
        self.log_text_edit.clear()
        try:
            with open(log_file_path, 'r') as log_file:
                logs = log_file.readlines()
                for log in logs:
                    self.log_text_edit.append(log.strip())
        except FileNotFoundError:
            self.log_text_edit.append(f"Tab: {self.tab_name}, Button: {name_label.text()}, 没有找到日志文件")


class MainWindow(QMainWindow):
    def __init__(self):
        super(MainWindow, self).__init__()
        self.setWindowTitle("FFing_点点")
        self.setGeometry(100, 100, 800, 600)

        # 确保 logs 目录存在
        logs_dir = resource_path("logs")
        if not os.path.exists(logs_dir):
            os.makedirs(logs_dir)



        # 初始化配置文件
        self.config_file = resource_path("conf.ini")
        self.config = configparser.ConfigParser()
        self.config.read(self.config_file)

        # 确保配置文件中有 'Tabs' 部分
        if 'Tabs' not in self.config:
            self.config['Tabs'] = {}
            # 创建默认选项卡
            for i in range(8):  # 假设我们有8个选项卡
                tab_name = f"选项卡{i + 1}"
                self.config.set("Tabs", f"tab{i + 1}_name", tab_name)
                self.config[tab_name] = {}
                for j in range(10):
                    self.config.set(tab_name, f"button{j + 1}_name", f"按钮{j + 1}")
                    self.config.set(tab_name, f"button{j + 1}_path", "echo '按钮被点击了'")
            with open(self.config_file, 'w') as configfile:
                self.config.write(configfile)

        # 创建选项卡控件
        self.tab_widget = QTabWidget(self)
        self.setCentralWidget(self.tab_widget)

        # 连接双击选项卡的信号到槽函数
        self.tab_widget.tabBar().setMovable(True)
        self.tab_widget.tabBar().tabBarDoubleClicked.connect(self.rename_tab)

        # 添加默认的选项卡
        for i in range(8):  # 假设我们有8个选项卡
            tab_name = self.config.get("Tabs", f"tab{i + 1}_name", fallback=f"选项卡{i + 1}")
            tab = ButtonPanel(tab_name)
            self.tab_widget.addTab(tab, tab_name)

    def rename_tab(self, index):
        if index >= 0:
            current_name = self.tab_widget.tabText(index)
            dialog = RenameTabDialog(current_name, self)

            if dialog.exec_() == QDialog.Accepted:
                new_name = dialog.get_data()

                # 更新选项卡名称
                self.tab_widget.setTabText(index, new_name)

                # 读取配置文件
                self.config = configparser.ConfigParser()
                self.config.read(self.config_file)

                # 检查旧选项卡是否存在配置信息
                if current_name in self.config:
                    # 如果新选项卡名称已经存在，则不需要复制旧的配置信息
                    if new_name not in self.config:
                        # 复制旧选项卡的配置信息到新选项卡
                        self.config[new_name] = self.config[current_name]
                        # 删除旧选项卡的配置信息
                        del self.config[current_name]

                # 更新 'Tabs' 部分中的选项卡名称
                self.config.set("Tabs", f"tab{index + 1}_name", new_name)

                # 将更新后的配置信息写入配置文件
                with open(self.config_file, 'w') as configfile:
                    self.config.write(configfile)

                # 重新加载按钮面板的配置信息
                tab = ButtonPanel(new_name)
                self.tab_widget.removeTab(index)
                self.tab_widget.insertTab(index, tab, new_name)


if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = MainWindow()
    window.show()
    sys.exit(app.exec_())
